import SwiftUI

struct LifeClockView: View {
    @StateObject private var viewModel = LifeClockViewModel()
    
    var body: some View {
        ScrollView {
            VStack(spacing: 16) {
                // Main life expectancy display
                VStack {
                    Text("Life Expectancy")
                        .font(.caption)
                        .foregroundColor(.secondary)
                    
                    Text(viewModel.formatLifeExpectancy())
                        .font(.title2)
                        .fontWeight(.bold)
                        .foregroundColor(.primary)
                }
                .padding()
                .background(Color.gray)
                .cornerRadius(12)
                
                // Status information
                VStack(alignment: .leading, spacing: 8) {
                    if viewModel.isLoading {
                        HStack {
                            ProgressView()
                                .scaleEffect(0.8)
                            Text("Updating...")
                                .font(.caption)
                        }
                    } else {
                        Text("Updated \(viewModel.formatLastUpdated())")
                            .font(.caption)
                            .foregroundColor(.secondary)
                    }
                    
                    if let error = viewModel.errorMessage {
                        Text(error)
                            .font(.caption)
                            .foregroundColor(.red)
                            .multilineTextAlignment(.leading)
                    }
                    
                    // Retry button for permission recovery
                    if viewModel.showPermissionRecoveryButton {
                        Button(action: {
                            Task {
                                await viewModel.retryPermissionRecovery()
                            }
                        }) {
                            Label("Retry", systemImage: "arrow.clockwise")
                                .font(.caption)
                        }
                        .buttonStyle(.bordered)
                        .controlSize(.small)
                    }
                }
                
                // Refresh button
                Button(action: {
                    Task {
                        await viewModel.updateLifeExpectancy()
                    }
                }) {
                    Label("Refresh", systemImage: "arrow.clockwise")
                        .font(.caption)
                }
                .buttonStyle(.bordered)
                .disabled(viewModel.isLoading)
                
                Spacer()
            }
            .padding()
        }
        .navigationTitle("LifeClock")
    }
}

#Preview {
    LifeClockView()
}
